from app.extensions import db
from app.models.water_asset import WaterAsset
from app.models.operator import Operator
from app.models.om_visit import OMVisit
from app.models.spare_part_inventory import SparePartInventory
from app.models.staff_dev_record import StaffDevelopmentRecord
from app.models.staff_dev_need import StaffDevelopmentNeed
from app.models.water_quality_test import WaterQualityTest
from app.models.water_quality_test_type import WaterQualityTestType
from app import create_app
from datetime import datetime, timezone, date, timedelta
from geoalchemy2 import WKTElement
from app.models.location import Location
from app.models.operational_status import OperationalStatus

def seed_garissa_data():
    app = create_app()
    with app.app_context():
        # Clear existing data (optional, depending on use case)
        # db.session.query(StaffDevelopmentRecord).delete()
        # db.session.query(SparePartInventory).delete()
        # db.session.query(OMVisit).delete()
        # db.session.query(Operator).delete()
        # db.session.query(WaterAsset).delete()
        # db.session.commit()

        # Query or create Location instance for Garissa
        location = db.session.query(Location).filter_by(county="Garissa").first()
        if not location:
            location = Location(county="Garissa", ward="Unknown", latitude=0.0, longitude=0.0)
            db.session.add(location)
            db.session.commit()

        # Query or create OperationalStatus instance for Active
        status_active = db.session.query(OperationalStatus).filter_by(name="Active").first()
        if not status_active:
            status_active = OperationalStatus(name="Active")
            db.session.add(status_active)
            db.session.commit()

        # Seed 10 water assets in Garissa
        assets = []
        for i in range(1, 11):
            asset = WaterAsset(
                name=f"Garissa Water Asset {i}",
                asset_type="Tank" if i % 3 == 0 else "Pump" if i % 3 == 1 else "Pipe",
                installation_date=datetime(2022, 1, i, 9, 0, 0, tzinfo=timezone.utc),
                material="Concrete" if i % 2 == 0 else "Steel",
                capacity=1000.0 + i * 100,
                location="Garissa",
                latitude=0.456 + i * 0.001,
                longitude=39.658 + i * 0.001,
                status="Active" if i % 2 == 0 else "Inactive",
                last_maintenance=datetime(2023, 5, i, 10, 0, 0, tzinfo=timezone.utc) if i % 2 == 0 else None
            )
            assets.append(asset)
            db.session.add(asset)
        db.session.commit()

        # Seed operators linked to water assets' water systems (assuming water_system_id is needed)
        # Since WaterAsset model does not show water_system_id, assuming operator links to water_systems
        # For this seed, we create dummy water_systems and link operators to them
        from app.models.water_system import WaterSystem
        water_systems = []
        for i in range(1, 6):
            ws = WaterSystem(
                unique_id=f"garissa_ws_{i}",
                name=f"Garissa Water System {i}",
                location=location,
                geom=WKTElement(f'POINT({39.65 + i * 0.002} {0.45 + i * 0.002})', srid=4326),
                status=status_active
            )
            water_systems.append(ws)
            db.session.add(ws)
        db.session.commit()

        operators = []
        for i, ws in enumerate(water_systems, start=1):
            op = Operator(
                water_system_id=ws.id,
                name=f"Operator {i}",
                training_supported=(i % 2 == 0),
                tariff_charged=(i % 2 == 1)
            )
            operators.append(op)
            db.session.add(op)
        db.session.commit()

        # Seed OM visits for water systems
        from app.models.om_visit_type import OMVisitType
        # Create some visit types if not exist
        visit_types = db.session.query(OMVisitType).all()
        if not visit_types:
            vt1 = OMVisitType(name="Routine")
            vt2 = OMVisitType(name="Emergency")
            db.session.add_all([vt1, vt2])
            db.session.commit()
            visit_types = [vt1, vt2]

        for i, ws in enumerate(water_systems, start=1):
            visit = OMVisit(
                water_system_id=ws.id,
                visit_type=visit_types[i % 2].id,
                visit_date=date.today() - timedelta(days=i*10),
                notes=f"OM visit notes for system {ws.name}"
            )
            db.session.add(visit)
        db.session.commit()

        # Seed spare parts inventory for water systems
        from app.models.spare_part_type import SparePartType
        # Create some spare part types if not exist
        spare_part_types = db.session.query(SparePartType).all()
        if not spare_part_types:
            spt1 = SparePartType(name="Valve")
            spt2 = SparePartType(name="Pump Seal")
            db.session.add_all([spt1, spt2])
            db.session.commit()
            spare_part_types = [spt1, spt2]

        for i, ws in enumerate(water_systems, start=1):
            inventory = SparePartInventory(
                water_system_id=ws.id,
                part_type_id=spare_part_types[i % 2].id,
                quantity=10 + i,
                last_restocked=date.today() - timedelta(days=i*5)
            )
            db.session.add(inventory)
        db.session.commit()

        # Seed staff development needs (create some if not exist)
        needs = db.session.query(StaffDevelopmentNeed).all()
        if not needs:
            need1 = StaffDevelopmentNeed(name="Safety Training")
            need2 = StaffDevelopmentNeed(name="Technical Skills")
            db.session.add_all([need1, need2])
            db.session.commit()
            needs = [need1, need2]

        # Seed staff development records linked to operators
        for i, op in enumerate(operators, start=1):
            record = StaffDevelopmentRecord(
                operator_id=op.id,
                need_id=needs[i % 2].id,
                is_met=(i % 2 == 0),
                date_met=date.today() - timedelta(days=i*15),
                date=date.today() - timedelta(days=i*30),
                notes=f"Staff development record for operator {op.name}"
            )
            db.session.add(record)
        db.session.commit()

        # Seed water quality test types (create some if not exist)
        test_types = db.session.query(WaterQualityTestType).all()
        if not test_types:
            wt1 = WaterQualityTestType(name="pH")
            wt2 = WaterQualityTestType(name="Turbidity")
            db.session.add_all([wt1, wt2])
            db.session.commit()
            test_types = [wt1, wt2]

        # Seed water quality tests linked to water systems
        for i, ws in enumerate(water_systems, start=1):
            test = WaterQualityTest(
                system_id=ws.id,
                test_type_id=test_types[i % 2].id,
                test_date=date.today() - timedelta(days=i*7),
                value=7.0 + i * 0.1
            )
            db.session.add(test)
        db.session.commit()

        print("Seeded Garissa water assets and related data successfully.")

if __name__ == "__main__":
    seed_garissa_data()
