# app/routes/monitoring_control_routes.py

from flask import Blueprint, request, jsonify
from app import db
from app.models.infrastructure_model import Infrastructure
from app.models.monitoring_model import MonitoringData
from app.models.alert_model import Alert

# Create a Blueprint for the Monitoring and Control Module
monitoring_control = Blueprint('monitoring_control', __name__)

# ----------------------------------------------------------
# 1. Create Real-Time Monitoring Data
# ----------------------------------------------------------

@monitoring_control.route('/monitoring/real_time', methods=['POST'])
def create_monitoring_data():
    data = request.get_json()

    # Create new monitoring data
    monitoring_data = MonitoringData(
        infrastructure_id=data['infrastructure_id'],
        flow_rate=data['flow_rate'],
        pressure=data['pressure'],
        water_level=data['water_level']
    )

    db.session.add(monitoring_data)
    db.session.commit()

    return jsonify({"message": "Real-time monitoring data logged successfully!"}), 201

# ----------------------------------------------------------
# 2. Retrieve Real-Time Monitoring Data for Infrastructure
# ----------------------------------------------------------

@monitoring_control.route('/monitoring/real_time/<int:infrastructure_id>', methods=['GET'])
def get_monitoring_data(infrastructure_id):
    monitoring_data = MonitoringData.query.filter_by(infrastructure_id=infrastructure_id).order_by(MonitoringData.timestamp.desc()).first()
    
    if not monitoring_data:
        return jsonify({"message": "No real-time data available for this infrastructure"}), 404

    return jsonify({
        "infrastructure_id": monitoring_data.infrastructure_id,
        "flow_rate": monitoring_data.flow_rate,
        "pressure": monitoring_data.pressure,
        "water_level": monitoring_data.water_level,
        "status": "Active",  # Example, can be extended to include actual status from infrastructure model
        "last_updated": monitoring_data.timestamp.isoformat()
    })

# ----------------------------------------------------------
# 3. Update Real-Time Monitoring Data
# ----------------------------------------------------------

@monitoring_control.route('/monitoring/real_time/<int:id>', methods=['PUT'])
def update_monitoring_data(id):
    monitoring_data = MonitoringData.query.get(id)

    if not monitoring_data:
        return jsonify({"message": "Monitoring data not found"}), 404

    data = request.get_json()

    # Update monitoring data fields
    monitoring_data.flow_rate = data.get('flow_rate', monitoring_data.flow_rate)
    monitoring_data.pressure = data.get('pressure', monitoring_data.pressure)
    monitoring_data.water_level = data.get('water_level', monitoring_data.water_level)

    db.session.commit()

    return jsonify({"message": "Real-time monitoring data updated successfully!"})

# ----------------------------------------------------------
# 4. Delete Real-Time Monitoring Data
# ----------------------------------------------------------

@monitoring_control.route('/monitoring/real_time/<int:id>', methods=['DELETE'])
def delete_monitoring_data(id):
    monitoring_data = MonitoringData.query.get(id)

    if not monitoring_data:
        return jsonify({"message": "Monitoring data not found"}), 404

    db.session.delete(monitoring_data)
    db.session.commit()

    return jsonify({"message": "Real-time monitoring data deleted successfully!"})

# ----------------------------------------------------------
# 5. Generate Alerts for System Issues (e.g., Low Pressure)
# ----------------------------------------------------------

@monitoring_control.route('/monitoring/alerts', methods=['POST'])
def generate_alert():
    data = request.get_json()

    # Create a new alert
    alert = Alert(
        alert_type=data['alert_type'],
        infrastructure_id=data['infrastructure_id'],
        message=data['message'],
        severity=data['severity']
    )

    db.session.add(alert)
    db.session.commit()

    return jsonify({"message": "Alert generated successfully!"}), 201

# ----------------------------------------------------------
# 6. Acknowledge or Resolve Alerts
# ----------------------------------------------------------

@monitoring_control.route('/monitoring/alerts/<int:alert_id>', methods=['PUT'])
def acknowledge_or_resolve_alert(alert_id):
    alert = Alert.query.get(alert_id)

    if not alert:
        return jsonify({"message": "Alert not found"}), 404

    data = request.get_json()

    alert.status = data.get('status', alert.status)
    db.session.commit()

    return jsonify({"message": "Alert status updated successfully!"})

# ----------------------------------------------------------
# 7. Control Infrastructure Components (Start/Stop/Reset)
# ----------------------------------------------------------

@monitoring_control.route('/monitoring/control/<int:infrastructure_id>', methods=['POST'])
def control_infrastructure(infrastructure_id):
    data = request.get_json()

    action = data.get('action')
    
    if action not in ['start', 'stop', 'reset']:
        return jsonify({"message": "Invalid action. Choose from 'start', 'stop', or 'reset'."}), 400

    infrastructure = Infrastructure.query.get(infrastructure_id)

    if not infrastructure:
        return jsonify({"message": "Infrastructure not found"}), 404

    # Example: Update infrastructure status or take action
    if action == 'start':
        infrastructure.status = 'Active'
    elif action == 'stop':
        infrastructure.status = 'Inactive'
    elif action == 'reset':
        infrastructure.status = 'Resetting'

    db.session.commit()

    return jsonify({"message": f"Infrastructure {action}d successfully!"})

# ----------------------------------------------------------
# 8. Retrieve System Alerts
# ----------------------------------------------------------

@monitoring_control.route('/monitoring/alerts', methods=['GET'])
def get_alerts():
    alerts = Alert.query.all()

    if not alerts:
        return jsonify({"message": "No alerts found."}), 404

    result = []
    for alert in alerts:
        result.append({
            "alert_type": alert.alert_type,
            "infrastructure_id": alert.infrastructure_id,
            "message": alert.message,
            "severity": alert.severity,
            "status": alert.status
        })

    return jsonify(result)

