# app/routes/mobile_routes.py

from flask import Blueprint, request, jsonify
from app import db
from app.models.task_model import Task
from app.models.data_model import DataSubmission
from app.models.user_model import User
from datetime import datetime
from werkzeug.utils import secure_filename
from app.models.notification_model import Notification


# Create a Blueprint for Mobile Application
mobile_routes = Blueprint('mobile_routes', __name__)

# Create a Blueprint for the Mobile Application Module
mobile_routes = Blueprint('mobile_routes', __name__)

# ----------------------------------------------------------
# 1. Create a Task (Mobile App - Field Assignment)
# ----------------------------------------------------------

@mobile_routes.route('/mobile/tasks', methods=['POST'])
def create_task():
    data = request.get_json()

    # Create a new task for field staff
    new_task = Task(
        task_name=data['task_name'],
        assigned_to=data['assigned_to'],
        location=data['location'],
        status='Pending',
        due_date=datetime.strptime(data['due_date'], '%Y-%m-%d'),
        description=data['description']
    )

    db.session.add(new_task)
    db.session.commit()

    return jsonify({"message": "Task created successfully!"}), 201

# ----------------------------------------------------------
# 2. Get All Tasks for a User (Read)
# ----------------------------------------------------------

@mobile_routes.route('/mobile/tasks', methods=['GET'])
def get_user_tasks():
    user_id = request.args.get('user_id')  # Assuming the user_id is passed as a query parameter
    tasks = Task.query.filter_by(assigned_to=user_id).all()

    result = [{"task_id": task.id, "task_name": task.task_name, "location": task.location, 
               "status": task.status, "due_date": task.due_date.isoformat(), "description": task.description}
              for task in tasks]

    return jsonify(result)

# ----------------------------------------------------------
# 3. Update Task Status (e.g., Completed, In Progress)
# ----------------------------------------------------------

@mobile_routes.route('/mobile/tasks/<int:id>', methods=['PUT'])
def update_task_status(id):
    task = Task.query.get(id)
    if not task:
        return jsonify({"message": "Task not found"}), 404

    data = request.get_json()
    
    # Update task status
    task.status = data.get('status', task.status)
    task.remarks = data.get('remarks', task.remarks)
    
    db.session.commit()

    return jsonify({"message": "Task status updated successfully!"})

# ----------------------------------------------------------
# 4. Delete Task (Mobile App - Task Removal)
# ----------------------------------------------------------

@mobile_routes.route('/mobile/tasks/<int:id>', methods=['DELETE'])
def delete_task(id):
    task = Task.query.get(id)
    
    if not task:
        return jsonify({"message": "Task not found"}), 404

    db.session.delete(task)
    db.session.commit()

    return jsonify({"message": "Task deleted successfully!"})

# ----------------------------------------------------------
# 5. Submit Data (Data Collection from the Field)
# ----------------------------------------------------------

@mobile_routes.route('/mobile/data_submission', methods=['POST'])
def submit_data():
    data = request.get_json()

    # Get task ID and form data
    task_id = data['task_id']
    form_data = data['form_data']  # Expecting a dictionary of question IDs and responses

    # Create a new data submission
    data_submission = DataSubmission(
        task_id=task_id,
        form_data=form_data,
        status='Pending',  # Initially set to 'Pending' until synced
        submitted_at=datetime.utcnow()
    )

    db.session.add(data_submission)
    db.session.commit()

    return jsonify({"message": "Data submitted successfully!"}), 201

# ----------------------------------------------------------
# 6. Get Submitted Data for a Task
# ----------------------------------------------------------

@mobile_routes.route('/mobile/data_submission/<int:task_id>', methods=['GET'])
def get_data_submission(task_id):
    data_submissions = DataSubmission.query.filter_by(task_id=task_id).all()

    result = [{"data_id": submission.id, "data": submission.data, 
               "status": submission.status, "submitted_at": submission.submitted_at.isoformat()}
              for submission in data_submissions]

    return jsonify(result)

# ----------------------------------------------------------
# 7. Sync Offline Data (Mobile App - Sync)
# ----------------------------------------------------------

@mobile_routes.route('/mobile/sync', methods=['POST'])
def sync_offline_data():
    data = request.get_json()

    # Sync all offline data for the user
    for task_data in data['offline_tasks']:
        task = Task.query.get(task_data['task_id'])
        
        if task:
            task.status = task_data['status']
            task.remarks = task_data.get('remarks', task.remarks)
            db.session.commit()

    for data_submission in data['offline_data']:
        submission = DataSubmission.query.get(data_submission['data_id'])
        
        if submission:
            submission.status = 'Synced'
            db.session.commit()

    return jsonify({"message": "Offline data synchronized successfully!"})

# ----------------------------------------------------------
# 8. Fetch User Profile (User Details)
# ----------------------------------------------------------

@mobile_routes.route('/mobile/profile', methods=['GET'])
def get_user_profile():
    user_id = request.args.get('user_id')
    user = User.query.get(user_id)

    if not user:
        return jsonify({"message": "User not found"}), 404

    return jsonify({
        "user_id": user.id,
        "username": user.username,
        "name": user.name,
        "role": user.role,
        "assigned_tasks": [task.id for task in user.tasks]
    })

# ----------------------------------------------------------
# 9. Update User Profile (Mobile App - Edit Profile)
# ----------------------------------------------------------

@mobile_routes.route('/mobile/profile/<int:id>', methods=['PUT'])
def update_user_profile(id):
    user = User.query.get(id)
    
    if not user:
        return jsonify({"message": "User not found"}), 404

    data = request.get_json()

    user.username = data.get('username', user.username)
    user.name = data.get('name', user.name)
    user.role = data.get('role', user.role)

    db.session.commit()

    return jsonify({"message": "User profile updated successfully!"})

@mobile_routes.route('/mobile/notifications', methods=['GET'])
def get_notifications():
    user_id = request.args.get('user_id')
    notifications = Notification.query.filter_by(user_id=user_id).all()

    result = [{"notification_id": notification.id, "message": notification.message, 
               "timestamp": notification.timestamp.isoformat()} 
              for notification in notifications]

    return jsonify(result)
    
 @mobile_routes.route('/mobile/location', methods=['GET'])
def get_location():
    user_id = request.args.get('user_id')
    user = User.query.get(user_id)

    if not user:
        return jsonify({"message": "User not found"}), 404

    # For simplicity, assume we get the user's current location
    return jsonify({"latitude": user.latitude, "longitude": user.longitude})


@mobile_routes.route('/mobile/media_upload', methods=['POST'])
def upload_media():
    if 'file' not in request.files:
        return jsonify({"message": "No file part"}), 400

    file = request.files['file']
    if file.filename == '':
        return jsonify({"message": "No selected file"}), 400

    filename = secure_filename(file.filename)
    file.save(os.path.join('uploads', filename))

    return jsonify({"message": "File uploaded successfully", "filename": filename}), 201
@mobile_routes.route('/mobile/task_history', methods=['GET'])
def get_task_history():
    user_id = request.args.get('user_id')
    task_history = Task.query.filter_by(assigned_to=user_id).all()

    history = [{
        "task_id": task.id,
        "task_name": task.task_name,
        "status": task.status,
        "due_date": task.due_date.isoformat(),
        "completed_at": task.completed_at.isoformat() if task.completed_at else None
    } for task in task_history]

    return jsonify(history)
@mobile_routes.route('/mobile/tasks/<int:id>/priority', methods=['POST'])
def set_task_priority(id):
    task = Task.query.get(id)
    if not task:
        return jsonify({"message": "Task not found"}), 404

    data = request.get_json()
    priority = data.get('priority', 'Low')  # Default to low priority

    task.priority = priority
    db.session.commit()

    # Notify the user of priority task
    send_notification(task.assigned_to, f"Priority task assigned: {task.task_name}")

    return jsonify({"message": "Task priority updated successfully!"})
@mobile_routes.route('/mobile/tasks/<int:id>/complete', methods=['POST'])
def complete_task(id):
    task = Task.query.get(id)
    
    if not task:
        return jsonify({"message": "Task not found"}), 404
    
    data = request.get_json()
    latitude = data.get('latitude')
    longitude = data.get('longitude')

    # Check if the worker is within the allowed geofenced area (e.g., 100 meters of task location)
    if not is_within_geofence(latitude, longitude, task.location):
        return jsonify({"message": "You must be within the task location to complete it."}), 400

    task.status = 'Completed'
    task.completed_at = datetime.utcnow()
    db.session.commit()

    return jsonify({"message": "Task marked as completed!"})
# app/routes/mobile_routes.py



# ----------------------------------------------------------
# 1. Create Notification (Task Assigned, Task Status Changed)
# ----------------------------------------------------------

@mobile_routes.route('/mobile/notifications', methods=['POST'])
def create_notification():
    data = request.get_json()

    # Ensure the user exists
    user = User.query.get(data['user_id'])
    if not user:
        return jsonify({"message": "User not found"}), 404

    # Create a new notification
    notification = Notification(
        user_id=data['user_id'],
        message=data['message'],
        notification_type=data['notification_type']
    )

    db.session.add(notification)
    db.session.commit()

    return jsonify({"message": "Notification created successfully!"}), 201
@mobile_routes.route('/mobile/notifications', methods=['GET'])
def get_notifications():
    user_id = request.args.get('user_id')  # User ID should be passed as query parameter
    notifications = Notification.query.filter_by(user_id=user_id).all()

    result = [{
        "notification_id": notification.id,
        "message": notification.message,
        "notification_type": notification.notification_type,
        "status": notification.status,
        "timestamp": notification.timestamp.isoformat()
    } for notification in notifications]

    return jsonify(result)
@mobile_routes.route('/mobile/notifications/<int:id>', methods=['PUT'])
def mark_notification_read(id):
    notification = Notification.query.get(id)

    if not notification:
        return jsonify({"message": "Notification not found"}), 404

    # Update the status of the notification to 'read'
    notification.status = 'read'
    db.session.commit()

    return jsonify({"message": "Notification marked as read successfully!"})
@mobile_routes.route('/mobile/notifications/unread', methods=['GET'])
def get_unread_notifications():
    user_id = request.args.get('user_id')
    unread_notifications = Notification.query.filter_by(user_id=user_id, status='unread').all()

    result = [{
        "notification_id": notification.id,
        "message": notification.message,
        "notification_type": notification.notification_type,
        "timestamp": notification.timestamp.isoformat()
    } for notification in unread_notifications]

    return jsonify(result)
@mobile_routes.route('/mobile/performance_metrics', methods=['GET'])
def get_performance_metrics():
    user_id = request.args.get('user_id')
    start_date = request.args.get('start_date', default='2023-01-01')
    end_date = request.args.get('end_date', default=datetime.utcnow().strftime('%Y-%m-%d'))

    # Convert string dates to datetime objects
    start_date = datetime.strptime(start_date, '%Y-%m-%d')
    end_date = datetime.strptime(end_date, '%Y-%m-%d')

    # Fetch tasks completed by the user within the specified date range
    tasks = Task.query.filter(
        Task.assigned_to == user_id,
        Task.created_at >= start_date,
        Task.created_at <= end_date
    ).all()

    # Track performance metrics for the tasks
    performance_metrics = PerformanceMetrics.query.filter(
        PerformanceMetrics.task_id.in_([task.id for task in tasks])
    ).all()

    # Calculate averages for performance metrics (e.g., average pressure, flow rate, water level)
    avg_pressure = sum([metric.pressure for metric in performance_metrics if metric.pressure]) / len(performance_metrics) if performance_metrics else 0
    avg_flow_rate = sum([metric.flow_rate for metric in performance_metrics if metric.flow_rate]) / len(performance_metrics) if performance_metrics else 0
    avg_water_level = sum([metric.water_level for metric in performance_metrics if metric.water_level]) / len(performance_metrics) if performance_metrics else 0

    return jsonify({
        "avg_pressure": avg_pressure,
        "avg_flow_rate": avg_flow_rate,
        "avg_water_level": avg_water_level
    })
# app/routes/mobile_routes.py

@mobile_routes.route('/mobile/tasks/<int:id>/priority', methods=['POST'])
def set_task_priority(id):
    task = Task.query.get(id)
    
    if not task:
        return jsonify({"message": "Task not found"}), 404

    data = request.get_json()
    priority = data.get('priority', 'Medium')  # Default to 'Medium' if no priority is provided

    # Update the task priority
    task.priority = priority
    db.session.commit()

    # Send push notification if task priority is High or Critical
    if task.priority in ['High', 'Critical']:
        send_push_notification(task.assigned_to, f"Priority task assigned: {task.task_name}")

    return jsonify({"message": "Task priority updated successfully!"})
@mobile_routes.route('/mobile/tasks/priority', methods=['GET'])
def get_high_priority_tasks():
    user_id = request.args.get('user_id')  # Get the user ID from the query parameters

    # Fetch tasks with priority 'High' or 'Critical'
    tasks = Task.query.filter(
        Task.assigned_to == user_id,
        Task.priority.in_(['High', 'Critical'])
    ).all()

    task_list = [{
        "task_id": task.id,
        "task_name": task.task_name,
        "priority": task.priority,
        "status": task.status,
        "due_date": task.due_date.isoformat() if task.due_date else None
    } for task in tasks]

    return jsonify({"high_priority_tasks": task_list})
@mobile_routes.route('/mobile/form/<task_type>', methods=['GET'])
def get_form(task_type):
    # Fetch form template for the given task type (e.g., 'maintenance', 'water_quality')
    form_fields = FormTemplate.query.filter_by(task_type=task_type).all()

    # Prepare the form field data for the response
    form_data = [{
        "id": field.id,
        "question": field.question,
        "field_type": field.field_type,
        "options": field.options.split(',') if field.options else [],
        "required": field.required
    } for field in form_fields]

    return jsonify({"form_fields": form_data})