from flask import Blueprint, request, jsonify
from app import db
from app.models.data_model import Data  # Assuming you have a Data model
from app.models.infrastructure_model import Infrastructure  # For infrastructure components
from app.models.performance_model import Performance  # For performance indicators
from app.models.maintenance_model import Maintenance  # For maintenance logs
from app.models.county_model import County  # Assuming you have a County model

# Create a Blueprint for the Data Management Module
data_routes = Blueprint('data_routes', __name__)

# ----------------------------------------------------------
# 1. Create Infrastructure Component (e.g., Pumps, Reservoirs)
# ----------------------------------------------------------

@data_routes.route('/infrastructure', methods=['POST'])
def create_infrastructure():
    data = request.get_json()

    # Create a new infrastructure component (Pump, Reservoir, etc.)
    new_infrastructure = Infrastructure(
        name=data['name'],
        type=data['type'],
        location=data['location'],
        status=data['status'],
        county=data['county'],
        capacity=data['capacity']
    )

    db.session.add(new_infrastructure)
    db.session.commit()

    return jsonify({"message": "Infrastructure component created successfully!"}), 201

# ----------------------------------------------------------
# 2. Retrieve All Infrastructure Components
# ----------------------------------------------------------

@data_routes.route('/infrastructure', methods=['GET'])
def get_infrastructure():
    infrastructures = Infrastructure.query.all()
    result = [{"id": infra.id, "name": infra.name, "type": infra.type, "location": infra.location, 
               "status": infra.status, "county": infra.county, "capacity": infra.capacity} for infra in infrastructures]
    
    return jsonify(result)

# ----------------------------------------------------------
# 3. Get Infrastructure by ID
# ----------------------------------------------------------

@data_routes.route('/infrastructure/<int:id>', methods=['GET'])
def get_infrastructure_by_id(id):
    infrastructure = Infrastructure.query.get(id)
    
    if infrastructure:
        return jsonify({
            "id": infrastructure.id,
            "name": infrastructure.name,
            "type": infrastructure.type,
            "location": infrastructure.location,
            "status": infrastructure.status,
            "county": infrastructure.county,
            "capacity": infrastructure.capacity
        })
    else:
        return jsonify({"message": "Infrastructure not found"}), 404

# ----------------------------------------------------------
# 4. Update Infrastructure Component
# ----------------------------------------------------------

@data_routes.route('/infrastructure/<int:id>', methods=['PUT'])
def update_infrastructure(id):
    infrastructure = Infrastructure.query.get(id)
    if not infrastructure:
        return jsonify({"message": "Infrastructure not found"}), 404

    data = request.get_json()

    # Update the infrastructure component details
    infrastructure.name = data.get('name', infrastructure.name)
    infrastructure.type = data.get('type', infrastructure.type)
    infrastructure.location = data.get('location', infrastructure.location)
    infrastructure.status = data.get('status', infrastructure.status)
    infrastructure.capacity = data.get('capacity', infrastructure.capacity)

    db.session.commit()

    return jsonify({"message": "Infrastructure component updated successfully!"})

# ----------------------------------------------------------
# 5. Delete Infrastructure Component
# ----------------------------------------------------------

@data_routes.route('/infrastructure/<int:id>', methods=['DELETE'])
def delete_infrastructure(id):
    infrastructure = Infrastructure.query.get(id)
    
    if not infrastructure:
        return jsonify({"message": "Infrastructure not found"}), 404

    db.session.delete(infrastructure)
    db.session.commit()

    return jsonify({"message": "Infrastructure component deleted successfully!"})

# ----------------------------------------------------------
# 6. Log Operational Data (e.g., Flow Rate, Pressure, etc.)
# ----------------------------------------------------------

@data_routes.route('/operational_data', methods=['POST'])
def log_operational_data():
    data = request.get_json()

    # Log operational data for a specific infrastructure component
    operational_data = Performance(
        infrastructure_id=data['infrastructure_id'],
        date=data['date'],
        flow_rate=data['flow_rate'],
        pressure=data['pressure'],
        water_level=data['water_level']
    )

    db.session.add(operational_data)
    db.session.commit()

    return jsonify({"message": "Operational data logged successfully!"}), 201

# ----------------------------------------------------------
# 7. Get Operational Data by Infrastructure ID
# ----------------------------------------------------------

@data_routes.route('/operational_data/<int:infrastructure_id>', methods=['GET'])
def get_operational_data(infrastructure_id):
    operational_data = Performance.query.filter_by(infrastructure_id=infrastructure_id).all()

    if not operational_data:
        return jsonify({"message": "No operational data found for this infrastructure"}), 404

    result = [{
        "date": data.date,
        "flow_rate": data.flow_rate,
        "pressure": data.pressure,
        "water_level": data.water_level
    } for data in operational_data]

    return jsonify(result)

# ----------------------------------------------------------
# 8. Get Performance Indicators for Infrastructure
# ----------------------------------------------------------

@data_routes.route('/performance_indicators/<int:infrastructure_id>', methods=['GET'])
def get_performance_indicators(infrastructure_id):
    performance = Performance.query.filter_by(infrastructure_id=infrastructure_id).first()

    if performance:
        return jsonify({
            "infrastructure_id": performance.infrastructure_id,
            "uptime": performance.uptime,
            "maintenance_due": performance.maintenance_due,
            "performance_rating": performance.performance_rating
        })
    else:
        return jsonify({"message": "Performance data not found"}), 404

# ----------------------------------------------------------
# 9. Fetch Historical Performance Report for a County
# ----------------------------------------------------------

@data_routes.route('/performance_report/<string:county>', methods=['GET'])
def get_performance_report(county):
    data = Data.query.filter_by(county=county).all()
    
    if not data:
        return jsonify({"message": "No data found for this county"}), 404

    result = [{"infrastructure_id": item.id, "name": item.name, "county": item.county, 
               "uptime": item.uptime, "maintenance_due": item.maintenance_due} for item in data]

    return jsonify(result)

# ----------------------------------------------------------
# 10. Log Maintenance Task (e.g., for pumps, reservoirs)
# ----------------------------------------------------------

@data_routes.route('/maintenance', methods=['POST'])
def log_maintenance():
    data = request.get_json()

    # Log maintenance tasks for infrastructure
    maintenance_task = Maintenance(
        infrastructure_id=data['infrastructure_id'],
        maintenance_date=data['maintenance_date'],
        description=data['description'],
        status=data['status']
    )

    db.session.add(maintenance_task)
    db.session.commit()

    return jsonify({"message": "Maintenance task logged successfully!"}), 201

# ----------------------------------------------------------
# 11. Get Maintenance History for Infrastructure
# ----------------------------------------------------------

@data_routes.route('/maintenance/<int:infrastructure_id>', methods=['GET'])
def get_maintenance_history(infrastructure_id):
    maintenance_tasks = Maintenance.query.filter_by(infrastructure_id=infrastructure_id).all()

    if not maintenance_tasks:
        return jsonify({"message": "No maintenance tasks found for this infrastructure"}), 404

    result = [{
        "maintenance_date": task.maintenance_date,
        "description": task.description,
        "status": task.status
    } for task in maintenance_tasks]

    return jsonify(result)
