from flask import Blueprint, render_template, request, jsonify, Response, redirect, url_for, flash, send_file
from app import db
from app.models.water_system import WaterSystem
from app.models.work_order import WorkOrder
from sqlalchemy import func
from werkzeug.utils import secure_filename
from io import StringIO, BytesIO
import pandas as pd
import numpy as np
import tensorflow as tf
import geojson
import csv
import plotly.graph_objects as go
from flask_login import login_required
from functools import wraps

gis_bp = Blueprint('gis', __name__)

ALLOWED_EXTENSIONS = {'csv'}

def allowed_file(filename):
    return '.' in filename and filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS

def login_required_role(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        return login_required(f)(*args, **kwargs)
    return decorated_function

@gis_bp.route('/')
@login_required
def index():
    assets = WaterSystem.query.all()
    return render_template('gis/index.html', assets=assets)

@gis_bp.route('/heatmap')
@login_required
def heatmap():
    return render_template('gis/heatmap.html')

@gis_bp.route('/draw')
@login_required
def draw_map():
    return render_template('gis/draw_map.html')

@gis_bp.route('/upload', methods=['GET', 'POST'])
@login_required
def upload():
    if request.method == 'POST':
        file = request.files.get('file')
        if not file or not allowed_file(file.filename):
            flash('Invalid file format', 'danger')
            return redirect(request.url)
        flash('File uploaded successfully', 'success')
        return redirect(url_for('gis.index'))
    return render_template('gis/upload.html')

@gis_bp.route('/export')
@login_required
def export_options():
    return render_template('gis/export_options.html')

@gis_bp.route('/regional_report')
@login_required
def regional_report_view():
    regional_data = db.session.query(
        func.substring(WaterSystem.name, 1, 2).label("region"),
        func.count(WaterSystem.id).label("count")
    ).group_by("region").all()
    return render_template('gis/regional_report.html', data=regional_data)

@gis_bp.route('/timeline')
@login_required
def timeline():
    return render_template('gis/timeline.html')

@gis_bp.route('/snapshot')
@login_required
def snapshot_export():
    return render_template('gis/snapshot_export.html')

@gis_bp.route('/api/assets')
def api_assets():
    assets = WaterSystem.query.all()
    return jsonify([
        {
            'id': a.id,
            'name': a.name,
            'type': a.type,
            'latitude': db.session.scalar(func.ST_Y(a.location)),
            'longitude': db.session.scalar(func.ST_X(a.location))
        } for a in assets
    ])

@gis_bp.route('/assets/create', methods=['GET', 'POST'])
def create_asset():
    if request.method == 'POST':
        name = request.form['name']
        asset_type = request.form['type']
        latitude = float(request.form['latitude'])
        longitude = float(request.form['longitude'])
        location = func.ST_SetSRID(func.ST_MakePoint(longitude, latitude), 4326)
        asset = WaterSystem(name=name, type=asset_type, location=location)
        db.session.add(asset)
        db.session.commit()
        flash('Asset created successfully.', 'success')
        return redirect(url_for('gis.index'))
    return render_template('gis/create.html')

@gis_bp.route('/assets/<int:asset_id>/edit', methods=['GET', 'POST'])
def edit_asset(asset_id):
    asset = WaterSystem.query.get_or_404(asset_id)
    if request.method == 'POST':
        asset.name = request.form['name']
        asset.type = request.form['type']
        lat = float(request.form['latitude'])
        lon = float(request.form['longitude'])
        asset.location = func.ST_SetSRID(func.ST_MakePoint(lon, lat), 4326)
        db.session.commit()
        flash('Asset updated successfully.', 'success')
        return redirect(url_for('gis.index'))
    return render_template('gis/edit.html', asset=asset)

@gis_bp.route('/assets/<int:asset_id>/delete', methods=['GET', 'POST'])
def delete_asset(asset_id):
    asset = WaterSystem.query.get_or_404(asset_id)
    if request.method == 'POST':
        db.session.delete(asset)
        db.session.commit()
        flash('Asset deleted successfully.', 'success')
        return redirect(url_for('gis.index'))
    return render_template('gis/delete.html', asset=asset)

@gis_bp.route('/nearby_assets', methods=['GET'])
def get_nearby_assets():
    try:
        lat = float(request.args.get('latitude'))
        lon = float(request.args.get('longitude'))
        radius = float(request.args.get('radius', 5000))
        nearby = WaterSystem.query.filter(
            func.ST_DWithin(
                WaterSystem.location,
                func.ST_MakePoint(lon, lat),
                radius
            )
        ).all()
        return jsonify([{
            "id": asset.id,
            "name": asset.name,
            "type": asset.type,
            "latitude": db.session.scalar(func.ST_Y(asset.location)),
            "longitude": db.session.scalar(func.ST_X(asset.location))
        } for asset in nearby])
    except Exception as e:
        return jsonify({"error": str(e)}), 400

@gis_bp.route('/api/gis/work_order_timeline')
def work_order_timeline():
    try:
        timeline = db.session.query(
            func.date(WorkOrder.created_at).label('date'),
            func.count(WorkOrder.id).label('count')
        ).group_by(func.date(WorkOrder.created_at)).order_by(func.date(WorkOrder.created_at)).all()
        return jsonify([{"date": str(row.date), "count": row.count} for row in timeline])
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@gis_bp.route('/api/gis/work_order_timeline_by_region')
def work_order_timeline_by_region():
    try:
        data = db.session.query(
            WorkOrder.region,
            func.date(WorkOrder.created_at).label('date'),
            func.count(WorkOrder.id)
        ).group_by(WorkOrder.region, func.date(WorkOrder.created_at)).order_by(func.date(WorkOrder.created_at)).all()
        output = {}
        for region, date, count in data:
            region = region or "Unknown"
            if region not in output:
                output[region] = []
            output[region].append({"date": str(date), "count": count})
        return jsonify(output)
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@gis_bp.route('/api/gis/dashboard_counts')
def dashboard_counts():
    try:
        total = db.session.query(func.count(WorkOrder.id)).scalar()
        open_tasks = db.session.query(func.count(WorkOrder.id)).filter(WorkOrder.status == 'Open').scalar()
        resolved_tasks = db.session.query(func.count(WorkOrder.id)).filter(WorkOrder.status == 'Resolved').scalar()
        return jsonify({"total": total, "open": open_tasks, "resolved": resolved_tasks})
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@gis_bp.route('/api/gis/timeline_status_breakdown')
def timeline_status_breakdown():
    try:
        data = db.session.query(
            func.date(WorkOrder.created_at).label('date'),
            WorkOrder.status,
            func.count(WorkOrder.id)
        ).group_by(func.date(WorkOrder.created_at), WorkOrder.status).order_by(func.date(WorkOrder.created_at)).all()
        grouped = {}
        for date, status, count in data:
            date = str(date)
            if date not in grouped:
                grouped[date] = {"Open": 0, "Resolved": 0}
            grouped[date][status] = count
        response = []
        for date in sorted(grouped):
            response.append({
                "date": date,
                "Open": grouped[date].get("Open", 0),
                "Resolved": grouped[date].get("Resolved", 0)
            })
        return jsonify(response)
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@gis_bp.route('/api/gis/asset_breakdown')
def asset_breakdown():
    try:
        by = request.args.get('by', 'county')
        if by == 'type':
            results = db.session.query(WaterSystem.type, func.count(WaterSystem.id)) \
                .group_by(WaterSystem.type).all()
            return jsonify([{"label": typ, "count": count} for typ, count in results])
        results = db.session.query(WaterSystem.county, func.count(WaterSystem.id)) \
            .group_by(WaterSystem.county).all()
        return jsonify([{"label": county, "count": count} for county, count in results])
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@gis_bp.route('/maintenance_hotspots')
def maintenance_hotspots():
    try:
        hotspots = db.session.query(
            WorkOrder.latitude,
            WorkOrder.longitude,
            func.count(WorkOrder.id)
        ).group_by(WorkOrder.latitude, WorkOrder.longitude).order_by(func.count(WorkOrder.id).desc()).all()
        return jsonify([{
            "latitude": lat,
            "longitude": lon,
            "count": count
        } for lat, lon, count in hotspots])
    except Exception as e:
        return jsonify({"error": str(e)}), 400
